<?php
session_start();
require_once '../includes/config.php';

// Verificar autenticación
if (!isLoggedIn()) {
    redirect('../login.php');
}

$empresa_id = getEmpresaActual();
$fecha_hoy = date('Y-m-d');
$fecha_ayer = date('Y-m-d', strtotime('-1 day'));

// Obtener estadísticas
try {
    // Total de empleados activos
    $total_empleados = $db->count(
        "SELECT COUNT(*) FROM empleados WHERE empresa_id = ? AND activo = 1", 
        [$empresa_id]
    );

    // Empleados presentes hoy
    $presentes_hoy = $db->count(
        "SELECT COUNT(*) FROM asistencias WHERE empresa_id = ? AND fecha = ? AND estado = 'presente'", 
        [$empresa_id, $fecha_hoy]
    );

    // Empleados ausentes hoy
    $ausentes_hoy = $db->count(
        "SELECT COUNT(*) FROM asistencias WHERE empresa_id = ? AND fecha = ? AND estado = 'ausente'", 
        [$empresa_id, $fecha_hoy]
    );

    // Tardanzas hoy
    $tardes_hoy = $db->count(
        "SELECT COUNT(*) FROM asistencias WHERE empresa_id = ? AND fecha = ? AND estado = 'tarde'", 
        [$empresa_id, $fecha_hoy]
    );

    // Proyectos activos
    $proyectos_activos = $db->count(
        "SELECT COUNT(*) FROM proyectos WHERE empresa_id = ? AND activo = 1", 
        [$empresa_id]
    );

    // Presentes ayer (para comparación)
    $presentes_ayer = $db->count(
        "SELECT COUNT(*) FROM asistencias WHERE empresa_id = ? AND fecha = ? AND estado = 'presente'", 
        [$empresa_id, $fecha_ayer]
    );

    // Calcular porcentaje de cambio
    $cambio_porcentaje = 0;
    if ($presentes_ayer > 0) {
        $cambio_porcentaje = (($presentes_hoy - $presentes_ayer) / $presentes_ayer) * 100;
    }

    // Actividad reciente
    $actividad_reciente = $db->fetchAll(
        "SELECT a.*, e.nombre, e.apellido, p.nombre as proyecto_nombre, u.nombre as registrado_por_nombre
         FROM asistencias a 
         INNER JOIN empleados e ON a.empleado_id = e.id 
         LEFT JOIN proyectos p ON a.proyecto_id = p.id
         INNER JOIN usuarios u ON a.registrado_por = u.id
         WHERE a.empresa_id = ? 
         ORDER BY a.fecha_registro DESC 
         LIMIT 10", 
        [$empresa_id]
    );

    // Datos para gráfico de los últimos 7 días
    $datos_grafico = [];
    for ($i = 6; $i >= 0; $i--) {
        $fecha = date('Y-m-d', strtotime("-$i days"));
        $presentes = $db->count(
            "SELECT COUNT(*) FROM asistencias WHERE empresa_id = ? AND fecha = ? AND estado = 'presente'", 
            [$empresa_id, $fecha]
        );
        $datos_grafico[] = [
            'fecha' => $fecha,
            'dia' => date('D', strtotime($fecha)),
            'presentes' => $presentes
        ];
    }

} catch (Exception $e) {
    logError('Error en dashboard: ' . $e->getMessage());
    $total_empleados = $presentes_hoy = $ausentes_hoy = $proyectos_activos = 0;
    $actividad_reciente = [];
    $cambio_porcentaje = 0;
}

$page_title = "Dashboard - Asistool";
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    
    <style>
        :root {
            --primary-color: #3b82f6;
            --secondary-color: #22c55e;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #06b6d4;
            --light-bg: #f8fafc;
            --card-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
        }

        body {
            background: var(--light-bg);
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            margin-left: 280px;
        }

        .main-content {
            padding: 2rem;
            min-height: 100vh;
        }

        .page-header {
            display: flex;
            justify-content: between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid #e5e7eb;
        }

        .page-title {
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-primary);
            margin: 0;
        }

        .header-actions {
            display: flex;
            gap: 0.75rem;
        }

        .btn-custom {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            font-weight: 500;
            border: none;
            transition: all 0.2s ease;
        }

        .btn-outline-primary {
            border: 1px solid var(--primary-color);
            color: var(--primary-color);
            background: white;
        }

        .btn-outline-primary:hover {
            background: var(--primary-color);
            color: white;
        }

        .btn-primary-custom {
            background: var(--primary-color);
            color: white;
        }

        .btn-primary-custom:hover {
            background: #2563eb;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: var(--card-shadow);
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
        }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }

        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
        }

        .stat-icon.primary { background: var(--primary-color); }
        .stat-icon.success { background: var(--success-color); }
        .stat-icon.warning { background: var(--warning-color); }
        .stat-icon.danger { background: var(--danger-color); }
        .stat-icon.info { background: var(--info-color); }

        .stat-title {
            font-size: 0.875rem;
            color: var(--text-secondary);
            margin: 0;
            font-weight: 500;
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-primary);
            margin: 0;
            line-height: 1;
        }

        .stat-change {
            display: flex;
            align-items: center;
            gap: 0.25rem;
            font-size: 0.875rem;
            margin-top: 0.5rem;
        }

        .stat-change.positive {
            color: var(--success-color);
        }

        .stat-change.negative {
            color: var(--danger-color);
        }

        .content-grid {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 2rem;
            margin-bottom: 2rem;
        }

        .card-custom {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: var(--card-shadow);
        }

        .card-header-custom {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 1px solid #e5e7eb;
        }

        .card-title {
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--text-primary);
            margin: 0;
        }

        .activity-item {
            display: flex;
            align-items: center;
            padding: 0.75rem 0;
            border-bottom: 1px solid #f3f4f6;
        }

        .activity-item:last-child {
            border-bottom: none;
        }

        .activity-icon {
            width: 32px;
            height: 32px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 0.75rem;
            font-size: 0.875rem;
            color: white;
        }

        .activity-content {
            flex: 1;
        }

        .activity-title {
            font-size: 0.875rem;
            font-weight: 500;
            color: var(--text-primary);
            margin: 0;
        }

        .activity-time {
            font-size: 0.75rem;
            color: var(--text-secondary);
            margin: 0;
        }

        .activity-badge {
            padding: 0.25rem 0.5rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .quick-actions {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }

        .quick-action-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: var(--card-shadow);
            text-align: center;
            transition: all 0.2s ease;
            text-decoration: none;
            color: inherit;
        }

        .quick-action-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
            text-decoration: none;
            color: inherit;
        }

        .quick-action-icon {
            width: 56px;
            height: 56px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            font-size: 1.5rem;
            color: white;
        }

        .quick-action-title {
            font-size: 1rem;
            font-weight: 600;
            color: var(--text-primary);
            margin: 0;
        }

        @media (max-width: 768px) {
            body {
                margin-left: 0;
            }
            
            .main-content {
                padding: 1rem;
            }
            
            .content-grid {
                grid-template-columns: 1fr;
            }
            
            .page-header {
                flex-direction: column;
                gap: 1rem;
                align-items: flex-start;
            }
        }

        .chart-container {
            position: relative;
            height: 300px;
            margin-top: 1rem;
        }

        .empty-state {
            text-align: center;
            padding: 2rem;
            color: var(--text-secondary);
        }

        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: var(--text-secondary);
        }
    </style>
</head>
<body>
    <!-- Incluir Sidebar -->
    <?php include 'sidebar.php'; ?>

    <!-- Contenido Principal -->
    <div class="main-content">
        <!-- Header de la Página -->
        <div class="page-header">
            <h1 class="page-title">Dashboard</h1>
            <div class="header-actions">
                <button class="btn btn-outline-primary btn-custom" onclick="location.reload()">
                    <i class="fas fa-sync-alt"></i> Actualizar
                </button>
                <a href="asistencia.php" class="btn btn-primary-custom btn-custom">
                    <i class="fas fa-plus"></i> Pasar Lista
                </a>
            </div>
        </div>

        <!-- Grid de Estadísticas -->
        <div class="stats-grid">
            <!-- Total Empleados -->
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <p class="stat-title">Total Empleados</p>
                        <h2 class="stat-value"><?= $total_empleados ?></h2>
                    </div>
                    <div class="stat-icon primary">
                        <i class="fas fa-users"></i>
                    </div>
                </div>
                <div class="stat-change">
                    <i class="fas fa-chart-line"></i>
                    <span>Empleados activos</span>
                </div>
            </div>

            <!-- Presentes Hoy -->
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <p class="stat-title">Presentes Hoy</p>
                        <h2 class="stat-value"><?= $presentes_hoy ?></h2>
                    </div>
                    <div class="stat-icon success">
                        <i class="fas fa-check-circle"></i>
                    </div>
                </div>
                <div class="stat-change <?= $cambio_porcentaje >= 0 ? 'positive' : 'negative' ?>">
                    <i class="fas fa-arrow-<?= $cambio_porcentaje >= 0 ? 'up' : 'down' ?>"></i>
                    <span><?= abs(round($cambio_porcentaje, 1)) ?>% vs ayer</span>
                </div>
            </div>

            <!-- Ausentes/Tardes Hoy -->
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <p class="stat-title">Ausentes + Tardes</p>
                        <h2 class="stat-value"><?= $ausentes_hoy + $tardes_hoy ?></h2>
                    </div>
                    <div class="stat-icon warning">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                </div>
                <div class="stat-change">
                    <i class="fas fa-users-slash"></i>
                    <span>Requieren atención</span>
                </div>
            </div>

            <!-- Proyectos Activos -->
            <div class="stat-card">
                <div class="stat-header">
                    <div>
                        <p class="stat-title">Proyectos Activos</p>
                        <h2 class="stat-value"><?= $proyectos_activos ?></h2>
                    </div>
                    <div class="stat-icon info">
                        <i class="fas fa-project-diagram"></i>
                    </div>
                </div>
                <div class="stat-change positive">
                    <i class="fas fa-plus"></i>
                    <span>En desarrollo</span>
                </div>
            </div>
        </div>

        <!-- Grid de Contenido -->
        <div class="content-grid">
            <!-- Gráfico de Asistencia -->
            <div class="card-custom">
                <div class="card-header-custom">
                    <h3 class="card-title">Asistencia Semanal</h3>
                    <a href="reportes.php" class="btn btn-sm btn-outline-primary">
                        Ver detalles <i class="fas fa-arrow-right"></i>
                    </a>
                </div>
                <div class="chart-container">
                    <canvas id="asistenciaChart"></canvas>
                </div>
            </div>

            <!-- Actividad Reciente -->
            <div class="card-custom">
                <div class="card-header-custom">
                    <h3 class="card-title">Actividad Reciente</h3>
                    <a href="#" class="btn btn-sm btn-outline-primary">
                        Ver todo <i class="fas fa-arrow-right"></i>
                    </a>
                </div>
                
                <?php if (empty($actividad_reciente)): ?>
                    <div class="empty-state">
                        <i class="fas fa-clock"></i>
                        <p>No hay actividad reciente</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($actividad_reciente as $item): ?>
                        <div class="activity-item">
                            <div class="activity-icon <?= $item['estado'] == 'presente' ? 'success' : ($item['estado'] == 'ausente' ? 'danger' : 'warning') ?>">
                                <i class="fas fa-<?= $item['estado'] == 'presente' ? 'check' : ($item['estado'] == 'ausente' ? 'times' : 'clock') ?>"></i>
                            </div>
                            <div class="activity-content">
                                <p class="activity-title">
                                    <?= htmlspecialchars($item['nombre'] . ' ' . $item['apellido']) ?>
                                </p>
                                <p class="activity-time">
                                    <?= formatFecha($item['fecha'], 'd/m/Y') ?> - 
                                    <?= formatFecha($item['fecha_registro'], 'H:i') ?>
                                </p>
                            </div>
                            <span class="activity-badge" style="background: <?= $item['estado'] == 'presente' ? '#dcfce7; color: #16a34a' : ($item['estado'] == 'ausente' ? '#fef2f2; color: #dc2626' : '#fef3c7; color: #d97706') ?>">
                                <?= ucfirst($item['estado']) ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Acceso Rápido -->
        <div class="card-custom">
            <div class="card-header-custom">
                <h3 class="card-title">Acceso Rápido</h3>
            </div>
            <div class="quick-actions">
                <a href="empleados.php" class="quick-action-card">
                    <div class="quick-action-icon primary">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <h4 class="quick-action-title">Gestionar Empleados</h4>
                </a>
                
                <a href="asistencia.php" class="quick-action-card">
                    <div class="quick-action-icon success">
                        <i class="fas fa-clipboard-check"></i>
                    </div>
                    <h4 class="quick-action-title">Pasar Lista Hoy</h4>
                </a>
                
                <a href="reportes.php" class="quick-action-card">
                    <div class="quick-action-icon info">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h4 class="quick-action-title">Ver Reportes</h4>
                </a>
                
                <a href="proyectos.php" class="quick-action-card">
                    <div class="quick-action-icon warning">
                        <i class="fas fa-project-diagram"></i>
                    </div>
                    <h4 class="quick-action-title">Gestionar Proyectos</h4>
                </a>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Configurar gráfico de asistencia
        const ctx = document.getElementById('asistenciaChart').getContext('2d');
        const datosGrafico = <?= json_encode($datos_grafico) ?>;
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: datosGrafico.map(d => d.dia),
                datasets: [{
                    label: 'Empleados Presentes',
                    data: datosGrafico.map(d => d.presentes),
                    borderColor: '#22c55e',
                    backgroundColor: 'rgba(34, 197, 94, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4,
                    pointBackgroundColor: '#22c55e',
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 2,
                    pointRadius: 6
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: '#f3f4f6'
                        },
                        ticks: {
                            color: '#6b7280'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        },
                        ticks: {
                            color: '#6b7280'
                        }
                    }
                },
                elements: {
                    point: {
                        hoverRadius: 8
                    }
                }
            }
        });

        // Auto-refresh cada 5 minutos
        setInterval(() => {
            if (document.visibilityState === 'visible') {
                location.reload();
            }
        }, 300000);
    </script>
</body>
</html>