<?php
/**
 * Configuración principal de la plataforma SaaS de Asistencia
 * @author Tu Nombre
 * @version 1.0
 */

// Iniciar sesión si no está iniciada
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// ====================================
// CONFIGURACIÓN DE BASE DE DATOS
// ====================================
define('DB_HOST', 'localhost');
define('DB_NAME', 'symcarpe_asistencia');
define('DB_USER', 'symcarpe_admin');
define('DB_PASS', 'Losmolles06');
define('DB_CHARSET', 'utf8mb4');

// ====================================
// CONFIGURACIÓN DE LA APLICACIÓN
// ====================================
define('APP_NAME', 'Asistool');
define('APP_VERSION', '1.0');
define('APP_URL', 'https://' . $_SERVER['HTTP_HOST']);
define('BASE_PATH', dirname(__DIR__));

// Configuración de zona horaria
date_default_timezone_set('America/Lima');

// ====================================
// CONFIGURACIÓN DE SEGURIDAD
// ====================================
define('ENCRYPT_KEY', 'asistool_2025_secret_key_change_in_production');
define('SESSION_TIMEOUT', 3600); // 1 hora en segundos
define('MAX_LOGIN_ATTEMPTS', 5);
define('LOGIN_LOCKOUT_TIME', 900); // 15 minutos

// ====================================
// CONFIGURACIÓN SAAS
// ====================================
define('SAAS_MODE', true);
define('DEFAULT_PLAN', 'basico');
define('MAX_EMPLOYEES_BASIC', 25);
define('MAX_EMPLOYEES_PREMIUM', 500);

// ====================================
// CONFIGURACIÓN DE ARCHIVOS
// ====================================
define('UPLOAD_DIR', BASE_PATH . '/uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_IMAGE_TYPES', ['jpg', 'jpeg', 'png', 'gif']);

// ====================================
// CONFIGURACIÓN DE ERRORES
// ====================================
define('PRODUCTION', false); // Cambiar a true en producción

if (PRODUCTION) {
    error_reporting(0);
    ini_set('display_errors', 0);
} else {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
}

// ====================================
// CLASE DE CONEXIÓN A BASE DE DATOS
// ====================================
class Database {
    private static $instance = null;
    private $connection;
    private $connectionError = false;
    
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];
            
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            $this->connectionError = true;
            $errorMsg = "Error de conexión: " . $e->getMessage();
            
            // Mostrar error de conexión amigable
            if (!PRODUCTION) {
                $this->showConnectionError($e->getMessage());
            } else {
                error_log($errorMsg);
                die("Servicio temporalmente no disponible.");
            }
        }
    }
    
    private function showConnectionError($error) {
        ?>
        <!DOCTYPE html>
        <html lang="es">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Error de Conexión - Asistool</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
            <style>
                body {
                    background: linear-gradient(135deg, #fee2e2 0%, #fecaca 50%, #fca5a5 100%);
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    font-family: 'Inter', sans-serif;
                }
                .error-container {
                    background: white;
                    padding: 3rem;
                    border-radius: 16px;
                    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
                    max-width: 600px;
                    text-align: center;
                }
                .logo {
                    color: #ef4444;
                    font-size: 2.5rem;
                    font-weight: 800;
                    margin-bottom: 1rem;
                }
                .error-code {
                    background: #fef2f2;
                    padding: 1rem;
                    border-radius: 8px;
                    font-family: 'Courier New', monospace;
                    font-size: 0.9rem;
                    text-align: left;
                    margin: 1rem 0;
                    border-left: 4px solid #ef4444;
                }
            </style>
        </head>
        <body>
            <div class="error-container">
                <div class="logo">⚠️ Asistool</div>
                <h2>Error de Conexión a la Base de Datos</h2>
                <p class="text-muted mb-4">No se pudo conectar a la base de datos. Verifica la configuración.</p>
                
                <div class="error-code">
                    <?= htmlspecialchars($error) ?>
                </div>
                
                <h5>Posibles soluciones:</h5>
                <ul class="text-start">
                    <li>Verifica que la base de datos <code>symcarpe_asistencia</code> existe</li>
                    <li>Confirma que el usuario <code>symcarpe_admin</code> tiene permisos</li>
                    <li>Revisa que la contraseña sea correcta: <code>Losmolles06</code></li>
                    <li>Asegúrate de que MySQL esté funcionando</li>
                </ul>
                
                <div class="mt-4">
                    <button onclick="location.reload()" class="btn btn-danger">
                        🔄 Reintentar Conexión
                    </button>
                </div>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        if ($this->connectionError) {
            return null;
        }
        return $this->connection;
    }
    
    // Método para verificar si la conexión está bien
    public function isConnected() {
        return !$this->connectionError && $this->connection !== null;
    }
    
    // Método para ejecutar consultas preparadas
    public function query($sql, $params = []) {
        if (!$this->isConnected()) {
            throw new Exception("No hay conexión a la base de datos");
        }
        
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("Error en consulta: " . $e->getMessage());
            throw $e;
        }
    }
    
    // Método para obtener un solo registro
    public function fetch($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetch();
    }
    
    // Método para obtener múltiples registros
    public function fetchAll($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchAll();
    }
    
    // Método para insertar y obtener ID
    public function insert($sql, $params = []) {
        $this->query($sql, $params);
        return $this->connection->lastInsertId();
    }
    
    // Método para contar registros
    public function count($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchColumn();
    }
}

// ====================================
// FUNCIONES AUXILIARES
// ====================================

/**
 * Función para sanitizar datos de entrada
 */
function sanitize($data) {
    if (is_array($data)) {
        return array_map('sanitize', $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * Función para validar email
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Función para generar hash de password
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Función para verificar password
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Función para generar token único
 */
function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Función para formatear fecha en español
 */
function formatFecha($fecha, $formato = 'd/m/Y') {
    if (!$fecha) return '-';
    
    $meses = [
        1 => 'Enero', 2 => 'Febrero', 3 => 'Marzo', 4 => 'Abril',
        5 => 'Mayo', 6 => 'Junio', 7 => 'Julio', 8 => 'Agosto',
        9 => 'Septiembre', 10 => 'Octubre', 11 => 'Noviembre', 12 => 'Diciembre'
    ];
    
    $dias = [
        1 => 'Lunes', 2 => 'Martes', 3 => 'Miércoles', 4 => 'Jueves',
        5 => 'Viernes', 6 => 'Sábado', 0 => 'Domingo'
    ];
    
    $timestamp = is_string($fecha) ? strtotime($fecha) : $fecha;
    
    if ($formato === 'completo') {
        $dia_semana = $dias[date('w', $timestamp)];
        $dia = date('d', $timestamp);
        $mes = $meses[date('n', $timestamp)];
        $año = date('Y', $timestamp);
        return "$dia_semana $dia de $mes de $año";
    }
    
    return date($formato, $timestamp);
}

/**
 * Función para obtener empresa actual del usuario
 */
function getEmpresaActual() {
    if (!isset($_SESSION['user_empresa_id'])) {
        return null;
    }
    return $_SESSION['user_empresa_id'];
}

/**
 * Función para verificar si usuario está autenticado
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && isset($_SESSION['user_empresa_id']);
}

/**
 * Función para verificar rol de usuario
 */
function hasRole($role) {
    if (!isLoggedIn()) return false;
    return $_SESSION['user_rol'] === $role || $_SESSION['user_rol'] === 'admin';
}

/**
 * Función para redireccionar
 */
function redirect($url, $permanent = false) {
    if ($permanent) {
        header("HTTP/1.1 301 Moved Permanently");
    }
    header("Location: $url");
    exit();
}

/**
 * Función para mostrar alertas con Bootstrap
 */
function showAlert($message, $type = 'info') {
    return "<div class='alert alert-$type alert-dismissible fade show' role='alert'>
                $message
                <button type='button' class='btn-close' data-bs-dismiss='alert'></button>
            </div>";
}

/**
 * Función para logging de errores
 */
function logError($message, $context = []) {
    $logDir = BASE_PATH . '/logs';
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logMessage = date('Y-m-d H:i:s') . " - " . $message;
    if (!empty($context)) {
        $logMessage .= " - Context: " . json_encode($context);
    }
    error_log($logMessage . PHP_EOL, 3, $logDir . '/app.log');
}

/**
 * Función para calcular horas trabajadas
 */
function calcularHorasTrabajadas($horaEntrada, $horaSalida) {
    if (!$horaEntrada || !$horaSalida) return 0;
    
    $entrada = strtotime($horaEntrada);
    $salida = strtotime($horaSalida);
    
    if ($salida <= $entrada) return 0;
    
    $diferencia = $salida - $entrada;
    return round($diferencia / 3600, 2); // Convertir a horas con 2 decimales
}

// ====================================
// INICIALIZACIÓN
// ====================================

// Crear directorio de logs si no existe
$logDir = BASE_PATH . '/logs';
if (!file_exists($logDir)) {
    mkdir($logDir, 0755, true);
}

// Crear directorio de uploads si no existe
$uploadDir = BASE_PATH . '/uploads';
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// ====================================
// AUTOLOAD DE CLASES (Opcional)
// ====================================
spl_autoload_register(function ($class_name) {
    $class_file = BASE_PATH . '/classes/' . $class_name . '.php';
    if (file_exists($class_file)) {
        require_once $class_file;
    }
});

// ====================================
// VARIABLES GLOBALES
// ====================================
try {
    $db = Database::getInstance();
    
    // Verificar conexión
    if (!$db->isConnected()) {
        throw new Exception("No se pudo establecer conexión con la base de datos");
    }
    
} catch (Exception $e) {
    logError('Error al inicializar base de datos: ' . $e->getMessage());
    if (!PRODUCTION) {
        die("Error de configuración de base de datos");
    }
}

// Configuraciones por defecto para la vista
$page_title = APP_NAME;
$current_page = basename($_SERVER['PHP_SELF'], '.php');

?>