<?php
/**
 * Instalador automático para Asistool
 * Este archivo verifica y crea toda la estructura necesaria
 */

$errors = [];
$success = [];
$warnings = [];

// Verificar si ya está instalado
if (file_exists('includes/config.php') && file_exists('admin/dashboard.php')) {
    header("Location: login.php");
    exit;
}

// Procesar instalación
if ($_POST && isset($_POST['install'])) {
    
    // 1. Crear directorios necesarios
    $directories = [
        'includes',
        'admin', 
        'logs',
        'uploads',
        'uploads/fotos',
        'classes',
        'assets',
        'assets/css',
        'assets/js',
        'assets/img'
    ];
    
    foreach ($directories as $dir) {
        if (!file_exists($dir)) {
            if (mkdir($dir, 0755, true)) {
                $success[] = "✅ Directorio '$dir' creado correctamente";
            } else {
                $errors[] = "❌ No se pudo crear el directorio '$dir'";
            }
        } else {
            $warnings[] = "⚠️ Directorio '$dir' ya existe";
        }
    }
    
    // 2. Crear archivo .htaccess
    $htaccess = '# Configuración para Asistool
RewriteEngine On

# Redirigir al login si no hay archivo especificado
DirectoryIndex login.php

# Configuraciones de seguridad
<Files "config.php">
    Order allow,deny
    Deny from all
</Files>

<Files "*.log">
    Order allow,deny
    Deny from all
</Files>

# Configurar errores PHP
php_flag display_errors Off
php_flag log_errors On

# Configuraciones de sesión
php_value session.cookie_httponly 1
php_value session.use_only_cookies 1

# Configurar zona horaria
php_value date.timezone "America/Lima"

# Aumentar límites si es necesario
php_value upload_max_filesize "10M"
php_value post_max_size "10M"
php_value memory_limit "128M"

# Prevenir acceso directo a ciertos archivos
<FilesMatch "\.(sql|bak|inc)$">
    Order allow,deny
    Deny from all
</FilesMatch>';

    if (file_put_contents('.htaccess', $htaccess)) {
        $success[] = "✅ Archivo .htaccess creado correctamente";
    } else {
        $errors[] = "❌ No se pudo crear .htaccess";
    }
    
    // 3. Crear archivo index.php
    $index = '<?php
// Archivo index.php - Redirección al login
header("Location: login.php");
exit();
?>';
    
    if (file_put_contents('index.php', $index)) {
        $success[] = "✅ Archivo index.php creado correctamente";
    }
    
    // 4. Verificar base de datos
    $db_host = $_POST['db_host'] ?? 'localhost';
    $db_name = $_POST['db_name'] ?? 'symcarpe_asistencia';
    $db_user = $_POST['db_user'] ?? 'symcarpe_admin';
    $db_pass = $_POST['db_pass'] ?? 'Losmolles06';
    
    try {
        $dsn = "mysql:host=$db_host;charset=utf8mb4";
        $pdo = new PDO($dsn, $db_user, $db_pass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Verificar si la base de datos existe
        $stmt = $pdo->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '$db_name'");
        if ($stmt->rowCount() > 0) {
            $success[] = "✅ Base de datos '$db_name' encontrada";
        } else {
            $warnings[] = "⚠️ Base de datos '$db_name' no existe - créala manualmente";
        }
        
    } catch (PDOException $e) {
        $errors[] = "❌ Error de conexión a BD: " . $e->getMessage();
    }
    
    // 5. Crear config.php con los datos ingresados
    $config_content = '<?php
/**
 * Configuración principal de Asistool
 * Generado automáticamente por el instalador
 */

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Configuración de base de datos
define("DB_HOST", "' . $db_host . '");
define("DB_NAME", "' . $db_name . '");
define("DB_USER", "' . $db_user . '");
define("DB_PASS", "' . $db_pass . '");
define("DB_CHARSET", "utf8mb4");

// Configuración de la aplicación
define("APP_NAME", "Asistool");
define("APP_VERSION", "1.0");
define("APP_URL", "https://" . $_SERVER["HTTP_HOST"]);
define("BASE_PATH", dirname(__DIR__));

date_default_timezone_set("America/Lima");

// Configuración de seguridad
define("ENCRYPT_KEY", "asistool_2025_secret_key_change_in_production");
define("SESSION_TIMEOUT", 3600);
define("MAX_LOGIN_ATTEMPTS", 5);
define("LOGIN_LOCKOUT_TIME", 900);

// Configuración SaaS
define("SAAS_MODE", true);
define("DEFAULT_PLAN", "basico");
define("MAX_EMPLOYEES_BASIC", 25);
define("MAX_EMPLOYEES_PREMIUM", 500);

// Configuración de archivos
define("UPLOAD_DIR", BASE_PATH . "/uploads/");
define("MAX_FILE_SIZE", 5 * 1024 * 1024);
define("ALLOWED_IMAGE_TYPES", ["jpg", "jpeg", "png", "gif"]);

// Configuración de errores
define("PRODUCTION", false);

if (PRODUCTION) {
    error_reporting(0);
    ini_set("display_errors", 0);
} else {
    error_reporting(E_ALL);
    ini_set("display_errors", 1);
}

// Clase de conexión a base de datos
class Database {
    private static $instance = null;
    private $connection;
    private $connectionError = false;
    
    private function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];
            
            $this->connection = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            $this->connectionError = true;
            error_log("Error de conexión: " . $e->getMessage());
            if (!PRODUCTION) {
                die("Error de conexión a la base de datos: " . $e->getMessage());
            }
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->connection;
    }
    
    public function isConnected() {
        return !$this->connectionError && $this->connection !== null;
    }
    
    public function query($sql, $params = []) {
        if (!$this->isConnected()) {
            throw new Exception("No hay conexión a la base de datos");
        }
        
        try {
            $stmt = $this->connection->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("Error en consulta: " . $e->getMessage());
            throw $e;
        }
    }
    
    public function fetch($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetch();
    }
    
    public function fetchAll($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchAll();
    }
    
    public function insert($sql, $params = []) {
        $this->query($sql, $params);
        return $this->connection->lastInsertId();
    }
    
    public function count($sql, $params = []) {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchColumn();
    }
}

// Funciones auxiliares
function sanitize($data) {
    if (is_array($data)) {
        return array_map("sanitize", $data);
    }
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, "UTF-8");
}

function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

function formatFecha($fecha, $formato = "d/m/Y") {
    if (!$fecha) return "-";
    $timestamp = is_string($fecha) ? strtotime($fecha) : $fecha;
    return date($formato, $timestamp);
}

function getEmpresaActual() {
    return $_SESSION["user_empresa_id"] ?? null;
}

function isLoggedIn() {
    return isset($_SESSION["user_id"]) && isset($_SESSION["user_empresa_id"]);
}

function hasRole($role) {
    if (!isLoggedIn()) return false;
    return $_SESSION["user_rol"] === $role || $_SESSION["user_rol"] === "admin";
}

function redirect($url, $permanent = false) {
    if ($permanent) {
        header("HTTP/1.1 301 Moved Permanently");
    }
    header("Location: $url");
    exit();
}

function showAlert($message, $type = "info") {
    return "<div class=\"alert alert-$type alert-dismissible fade show\" role=\"alert\">
                $message
                <button type=\"button\" class=\"btn-close\" data-bs-dismiss=\"alert\"></button>
            </div>";
}

function logError($message, $context = []) {
    $logDir = BASE_PATH . "/logs";
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logMessage = date("Y-m-d H:i:s") . " - " . $message;
    if (!empty($context)) {
        $logMessage .= " - Context: " . json_encode($context);
    }
    error_log($logMessage . PHP_EOL, 3, $logDir . "/app.log");
}

function calcularHorasTrabajadas($horaEntrada, $horaSalida) {
    if (!$horaEntrada || !$horaSalida) return 0;
    
    $entrada = strtotime($horaEntrada);
    $salida = strtotime($horaSalida);
    
    if ($salida <= $entrada) return 0;
    
    $diferencia = $salida - $entrada;
    return round($diferencia / 3600, 2);
}

// Inicialización
$logDir = BASE_PATH . "/logs";
if (!file_exists($logDir)) {
    mkdir($logDir, 0755, true);
}

$uploadDir = BASE_PATH . "/uploads";
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Variables globales
try {
    $db = Database::getInstance();
    if (!$db->isConnected()) {
        throw new Exception("No se pudo establecer conexión con la base de datos");
    }
} catch (Exception $e) {
    logError("Error al inicializar base de datos: " . $e->getMessage());
    if (!PRODUCTION) {
        die("Error de configuración de base de datos");
    }
}

$page_title = APP_NAME;
$current_page = basename($_SERVER["PHP_SELF"], ".php");
?>';

    if (file_put_contents('includes/config.php', $config_content)) {
        $success[] = "✅ Archivo config.php creado correctamente";
    } else {
        $errors[] = "❌ No se pudo crear config.php";
    }
    
    // 6. Si todo fue exitoso, mostrar mensaje final
    if (empty($errors)) {
        $install_success = true;
    }
}

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalador - Asistool</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #dcfce7 0%, #bbf7d0 50%, #86efac 100%);
            min-height: 100vh;
            font-family: 'Inter', sans-serif;
        }
        .installer-container {
            max-width: 800px;
            margin: 2rem auto;
            padding: 0 1rem;
        }
        .installer-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .installer-header {
            background: #22c55e;
            color: white;
            padding: 2rem;
            text-align: center;
        }
        .installer-logo {
            font-size: 3rem;
            font-weight: 800;
            margin-bottom: 0.5rem;
        }
        .installer-body {
            padding: 2rem;
        }
        .step-indicator {
            display: flex;
            justify-content: center;
            margin-bottom: 2rem;
        }
        .step {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e5e7eb;
            color: #6b7280;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 0.5rem;
            font-weight: bold;
        }
        .step.active {
            background: #22c55e;
            color: white;
        }
        .step.completed {
            background: #10b981;
            color: white;
        }
        .message-group {
            margin-bottom: 1.5rem;
        }
        .message {
            padding: 0.75rem 1rem;
            border-radius: 8px;
            margin-bottom: 0.5rem;
        }
        .message.success {
            background: #f0fdf4;
            color: #16a34a;
            border-left: 4px solid #22c55e;
        }
        .message.error {
            background: #fef2f2;
            color: #dc2626;
            border-left: 4px solid #ef4444;
        }
        .message.warning {
            background: #fffbeb;
            color: #d97706;
            border-left: 4px solid #f59e0b;
        }
        .form-group {
            margin-bottom: 1.5rem;
        }
        .form-label {
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        .form-control {
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            padding: 0.75rem;
        }
        .form-control:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
        .btn-install {
            background: #22c55e;
            border: none;
            color: white;
            padding: 1rem 2rem;
            border-radius: 8px;
            font-weight: 600;
            width: 100%;
        }
        .btn-install:hover {
            background: #16a34a;
        }
        .final-actions {
            display: flex;
            gap: 1rem;
            margin-top: 2rem;
        }
        .final-actions .btn {
            flex: 1;
        }
    </style>
</head>
<body>
    <div class="installer-container">
        <div class="installer-card">
            <div class="installer-header">
                <div class="installer-logo">🚀 Asistool</div>
                <h2>Instalador Automático</h2>
                <p>Configuración rápida y sencilla</p>
            </div>
            
            <div class="installer-body">
                <?php if (isset($install_success) && $install_success): ?>
                    <!-- Instalación Completada -->
                    <div class="text-center">
                        <div style="font-size: 4rem; color: #22c55e;">🎉</div>
                        <h3>¡Instalación Completada!</h3>
                        <p class="text-muted">Asistool se ha instalado correctamente en tu servidor.</p>
                    </div>
                    
                    <?php if (!empty($success)): ?>
                        <div class="message-group">
                            <h5>✅ Tareas Completadas:</h5>
                            <?php foreach ($success as $msg): ?>
                                <div class="message success"><?= $msg ?></div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($warnings)): ?>
                        <div class="message-group">
                            <h5>⚠️ Advertencias:</h5>
                            <?php foreach ($warnings as $msg): ?>
                                <div class="message warning"><?= $msg ?></div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="alert alert-info">
                        <h6>📋 Próximos Pasos:</h6>
                        <ol>
                            <li>Elimina este archivo <code>install.php</code> por seguridad</li>
                            <li>Crea la base de datos <code>symcarpe_asistencia</code> si no existe</li>
                            <li>Ejecuta el script SQL para crear las tablas</li>
                            <li>Sube los archivos restantes (login.php, admin/, etc.)</li>
                        </ol>
                    </div>
                    
                    <div class="final-actions">
                        <a href="login.php" class="btn btn-success">
                            <i class="fas fa-sign-in-alt"></i> Ir al Login
                        </a>
                        <button class="btn btn-danger" onclick="if(confirm('¿Estás seguro?')) window.location.href='?delete=1'">
                            <i class="fas fa-trash"></i> Eliminar Instalador
                        </button>
                    </div>
                    
                <?php else: ?>
                    <!-- Formulario de Instalación -->
                    <div class="step-indicator">
                        <div class="step active">1</div>
                        <div class="step">2</div>
                        <div class="step">3</div>
                    </div>
                    
                    <?php if (!empty($errors)): ?>
                        <div class="message-group">
                            <h5>❌ Errores Encontrados:</h5>
                            <?php foreach ($errors as $error): ?>
                                <div class="message error"><?= $error ?></div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST">
                        <h4>Configuración de Base de Datos</h4>
                        <p class="text-muted">Ingresa los datos de conexión a tu base de datos MySQL</p>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Servidor (Host)</label>
                                    <input type="text" name="db_host" class="form-control" 
                                           value="<?= $_POST['db_host'] ?? 'localhost' ?>" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Nombre de la Base de Datos</label>
                                    <input type="text" name="db_name" class="form-control" 
                                           value="<?= $_POST['db_name'] ?? 'symcarpe_asistencia' ?>" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Usuario de la Base de Datos</label>
                                    <input type="text" name="db_user" class="form-control" 
                                           value="<?= $_POST['db_user'] ?? 'symcarpe_admin' ?>" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label class="form-label">Contraseña</label>
                                    <input type="password" name="db_pass" class="form-control" 
                                           value="<?= $_POST['db_pass'] ?? 'Losmolles06' ?>" required>
                                </div>
                            </div>
                        </div>
                        
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle"></i>
                            <strong>Importante:</strong> Asegúrate de que la base de datos existe y el usuario tiene permisos completos.
                        </div>
                        
                        <button type="submit" name="install" value="1" class="btn-install">
                            <i class="fas fa-rocket"></i> Instalar Asistool
                        </button>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <?php if (isset($_GET['delete'])): ?>
        <?php unlink(__FILE__); ?>
        <script>
            alert('Instalador eliminado correctamente');
            window.location.href = 'login.php';
        </script>
    <?php endif; ?>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>